package sim.portrayal;
import java.awt.*;
import sim.engine.*;
import sim.util.gui.*;
import sim.util.*;
import sim.display.*;
import javax.swing.*;

/**
   A simple inspector class that looks at the "getX" and "setX" method of the object to be investigates
   and creates a user-friendly graphical interface with read only and read/write components of the object.
*/

public class SimpleInspector extends Inspector
    {
    public static final int MAX_PROPERTIES = 20;
    public GUIState state;
    public Object object;
    public LabelledList propertyList;
    public Properties properties;
    public PropertyField[] members = null;
    public String name;
    public JPanel extraValue;
    public NumberTextField extraIndex;
    public int currentExtraIndex;
    public int len;  // the first N elements shown
    
    public SimpleInspector(Object object, GUIState state)
        {
        this(object,state,null);
        }
    
    public SimpleInspector(Object object, GUIState state, String name) 
        { 
        super();
        setLayout(new BorderLayout());
        this.object = object;
        this.state = state;
        this.name = name;
        generateProperties();
        }
    
    void setExtraPropertyValue()
        {
        if (extraIndex!=null)
            {
            extraValue.removeAll();
            extraValue.add(new JLabel(""+properties.getName(currentExtraIndex)+ "  "), BorderLayout.WEST);
            PropertyField newField = makePropertyField(currentExtraIndex);
            extraValue.add(newField, BorderLayout.CENTER);
            extraValue.repaint();
            }
        }
    
    PropertyField makePropertyField(final int index)
        {
        Class type = properties.getType(index);
        return new PropertyField(
            null,
            properties.betterToString(properties.getValue(index)),
            properties.isReadWrite(index),
            properties.getDomain(index),
            (properties.isComposite(index) ?
             PropertyField.SHOW_VIEWBUTTON : 
             (type == Boolean.TYPE || type == Boolean.class ?
              PropertyField.SHOW_CHECKBOX :
              (properties.getDomain(index) == null ? PropertyField.SHOW_TEXTFIELD :
               (properties.getDomain(index) instanceof Interval) ? 
               PropertyField.SHOW_SLIDER : PropertyField.SHOW_LIST ))))
            {
            Properties props = properties;

            // The return value should be the value you want the display to show instead.
            public String newValue(final String newValue)
                {
                final String retval[] = new String[1];  // man, anonymous classes are idiotic
                // compared to true closures...
                // the underlying model could still be running, so we need
                // to do this safely
                synchronized(SimpleInspector.this.state.state.schedule)
                    {
                    // try to set the value
                    if (props.setValue(index, newValue) == null)
                        java.awt.Toolkit.getDefaultToolkit().beep();
                    // refresh the controller
                    SimpleInspector.this.state.controller.refresh();
                    // set text to the new value
                    return props.getValue(index).toString();
                    }
                }

            public void viewProperty()
                {
                final SimpleInspector simpleInspector = new SimpleInspector(props.getValue(index), SimpleInspector.this.state);
                JScrollPane inspectorPane = new JScrollPane(simpleInspector);
                inspectorPane.setBorder(BorderFactory.createEmptyBorder(2,2,2,2));

                Steppable stepper = new Steppable()
                    {
                    public void step(final SimState state)
                        {
                        SwingUtilities.invokeLater(new Runnable()
                            {
                            Inspector inspector = simpleInspector;
                            public void run()
                                {
                                synchronized(state.schedule)
                                    {
                                    inspector.updateInspector();
                                    inspector.repaint();
                                    }
                                }
                            });
                        }
                    };
                                    
                final Stoppable stopper = SimpleInspector.this.state.scheduleImmediateRepeat(true,stepper);
                // put in new frame which stops when closed
                JFrame frame = new JFrame("" + props.getValue(index))
                    {
                    public void dispose()
                        {
                        super.dispose();
                        stopper.stop();
                        }
                    };

                frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
                frame.getContentPane().setLayout(new BorderLayout());
                frame.getContentPane().add(inspectorPane, BorderLayout.CENTER);
                frame.setResizable(true);
                frame.pack();
                frame.setVisible(true);
                }
            };
        }
    
    /** Private method.  Does a repaint that is guaranteed to work (on some systems, plain repaint())
        fails if there's lots of updates going on as is the case in our simulator thread.  */
    void doEnsuredRepaint(final Component component)
        {
        SwingUtilities.invokeLater(new Runnable()
            {
            public void run()
                {
                if (component!=null) component.repaint();
                }
            });
        }

    void generateProperties()
        {
        properties = Properties.getProperties(object,true,true,false,true);

        if (name==null) propertyList = new LabelledList();
        else propertyList = new LabelledList(name);

        len = properties.numProperties();
        
        // deal with the situation where you have more properties than you can show
        if (len > MAX_PROPERTIES)
            {
            len = MAX_PROPERTIES;
            if (name == null) name = "";
            name = name + " (First " + MAX_PROPERTIES + " of " + properties.numProperties() + ")";
            currentExtraIndex = len;
            extraIndex = new NumberTextField(currentExtraIndex,1,1)
                {
                public double newValue(double newValue)
                    {
                    currentExtraIndex = (int) newValue;
                    if (currentExtraIndex<0 || currentExtraIndex >= properties.numProperties())
                        currentExtraIndex = (int)getValue();
                    setExtraPropertyValue();
                    return currentExtraIndex;
                    }
                };
            extraIndex.valField.setColumns(4);
            extraValue = new JPanel();
            extraValue.setLayout(new BorderLayout());
            setExtraPropertyValue();
            propertyList.add(extraIndex,extraValue);
            }
            
    
        members = new PropertyField[len];

        for( int i = 0 ; i < len; i++ )
            {
            members[i] = makePropertyField(i);
            propertyList.addLabelled( properties.getName(i) + " ", members[i] );
            }
        add(propertyList, BorderLayout.CENTER);
        }
    
    public void updateInspector()
        {
        if (properties.isVolatile())  // need to rebuild each time, YUCK
            {
            remove(propertyList);
            generateProperties();
            revalidate();
            doEnsuredRepaint(this);
            }
        else for( int i = 0 ; i < len ; i++ )
            members[i].setValue( properties.betterToString(properties.getValue(i)));
        setExtraPropertyValue();
        }
    }
