package sim.app.schelling;
import sim.engine.*;
import sim.field.grid.*;
import sim.util.*;
import ec.util.*;
import java.io.*;

public /*strictfp*/ class Schelling extends SimState
    {
    public int gridHeight;
    public int gridWidth;
    public int neighborhood = 1;
    public int threshold = 3;
    public double redProbability = 0.3;
    public double blueProbability = 0.3;
    public double emptyProbability = 0.3;
    public double unavailableProbability = 0.1;

    // we presume that no one relies on these DURING a simulation
    public int getGridHeight() { return gridHeight; }
    public void setGridHeight(int val) { if (val > 0) gridHeight = val; }
    public int getGridWidth() { return gridWidth; }
    public void setGridWidth(int val) { if (val > 0) gridWidth = val; }
    public int getNeighborhood() { return neighborhood; }
    public void setNeighborhood(int val) { if (val > 0) neighborhood = val; }
    public int getThreshold() { return threshold; }
    public void setThreshold(int val) { if (val >= 0) threshold = val; }

    // some cutsie-pie probability sliders.  More work than necessary, but it was fun.
    public Object domRedProbability() { return new Interval(0.0,1.0); }
    public double getRedProbability() { return redProbability; }
    public void setRedProbability(double val) 
        { 
        if (val >= 0 && val <= 1.0)
            {
            redProbability = val;
            }
        }
    
    public Object domBlueProbability() { return new Interval(0.0,1.0); }
    public double getBlueProbability() { return blueProbability; }
    public void setBlueProbability(double val) 
        { 
        if (val >= 0 && val <= 1.0)
            {
            blueProbability = val;
            }
        }

    public Object domEmptyProbability() { return new Interval(0.0,1.0); }
    public double getEmptyProbability() { return emptyProbability; }
    public void setEmptyProbability(double val) 
        { 
        if (val >= 0 && val <= 1.0)
            {
            emptyProbability = val;
            }
        }

    public Object domUnavailableProbability() { return new Interval(0.0,1.0); }
    public double getUnavailableProbability() { return unavailableProbability; }
    public void setUnavailableProbability(double val) 
        { 
        if (val >= 0 && val <= 1.0)
            {
            unavailableProbability = val;
            double total = redProbability + blueProbability + emptyProbability; 
            if (total==0.0) total = 1.0;
            redProbability *= (1.0 - unavailableProbability)/total;
            blueProbability *= (1.0 - unavailableProbability)/total;
            emptyProbability *= (1.0 - unavailableProbability)/total;
            }
        }

    public IntGrid2D neighbors;
    public Bag emptySpaces = new Bag();
    public static final  int EMPTY = 0;
    public static final int UNAVAILABLE = 1;
    public static final int RED = 2;
    public static final int BLUE = 3;

    /** Creates a Schelling simulation with the given random number seed. */
    public Schelling(long seed)
        {
        this(seed, 100, 100);
        }
        
    public Schelling(long seed, int width, int height)
        {
        super(new MersenneTwisterFast(seed), new Schedule(1));
        gridWidth = width; gridHeight = height;
        createGrids();
        }

    protected void createGrids()
        {
        emptySpaces.clear();
        neighbors = new IntGrid2D(gridWidth, gridHeight,0);
        int[][] g = neighbors.field;
        for(int x=0;x<gridWidth;x++)
            for(int y=0;y<gridHeight;y++)
                {
                double d = random.nextDouble();
                if (d < redProbability) g[x][y] = RED;
                else if (d < redProbability + blueProbability) g[x][y] = BLUE;
                else if (d < redProbability + blueProbability + emptyProbability) 
                    { g[x][y] = EMPTY; emptySpaces.add(new Int2D(x,y)); }
                else g[x][y] = UNAVAILABLE;
                }
        }
    
    /** Resets and starts a simulation */
    public void start()
        {
        super.start();  // clear out the schedule
        
        // make new grids
        createGrids();
        for(int x=0;x<gridWidth;x++)
            for(int y=0;y<gridHeight;y++)
                {
                schedule.scheduleRepeating(new Agent(x,y));
                }
        }
    
    /** Runs Schelling headless (no GUI).  Checkpoints out to files every 500 simulation timesteps.
        If no command-line arguments are provided, a new Schelling is started.
        Optionally the user can say "-checkpoint [filename]" and Schelling will load a checkpoint
        and continue from that point. */
        
    public static void main(String[] args)
        {
        Schelling schelling = null;
        
        // should we load from checkpoint?  I wrote this little chunk of code to
        // check for this to give you the general idea.
        
        for(int x=0;x<args.length-1;x++)  // "-checkpoint" can't be the last string
            if (args[x].equals("-checkpoint"))
                {
                SimState state = SimState.readFromCheckpoint(new File(args[x+1]));
                if (state == null)   // there was an error -- it got printed out to the screen, so just quit
                    System.exit(1);
                else if (!(state instanceof Schelling))  // uh oh, wrong simulation stored in the file!
                    {
                    System.out.println("Checkpoint contains some other simulation: " + state);
                    System.exit(1);
                    }
                else // we're ready to lock and load!  
                    schelling = (Schelling)state;
                }
        
        // ...or should we start fresh?
        if (schelling==null)  // no checkpoint file requested
            {
            schelling = new Schelling(System.currentTimeMillis());  // make a new schelling.  Seed the RNG with the time
            schelling.start();  // prep the bugs!
            System.out.println("Starting Schelling.  Running for 5000 steps.");
            }
        
        long time;
        while((time = schelling.schedule.time()) < 5000)
            {
            // step the schedule.  This is where everything happens.
            if (!schelling.schedule.step(schelling)) break;   // it won't happen that we end prematurely,
            // but it's worth checking for!

            if (time%100==0 && time!=0) 
                System.out.println("Time Step " + time);
        
            // checkpoint
            if (time%500==0 && time!=0)
                {
                String s = "hb." + time + ".checkpoint";
                System.out.println("Checkpointing to file: " + s);
                schelling.writeToCheckpoint(new File(s));
                }
            }
            
        schelling.finish();  // we don't use this, but it's good style
                
        // If we're using the ThreadedDiffuser, it uses ParallelSequence, and so we have to 
        // explicitly say System.exit(0) here instead of just dropping out -- see the docs
        // on ParallelSequence for more information.
        System.exit(0);
        }
        

    }
    
    
    
    
    
