#include <stdio.h>	
#include <unistd.h>	
#include <sys/types.h>	
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <netdb.h>
#include <linux/limits.h>



//gcc -Wall ruminate_client.c -o ~/bin/ruminate_client
/*

The vortex protocol is a protocol for passing data including variable length data and a variable length metadata

The protocol is defined as follows:

The vortex protocol allows a producer to provide network stream data to a consumer. The network data is accompanied by metadata. Both are arbitrary and are limited only by the size of the length fields in the protocol (32bits which can specify up to 4 GB). The consumer requests a number of streams of the producer and the producer provides them. 
The client can request streams using a single byte request from 1 - 255. The producer and consumer must agree (out of band) on the conventions how the request number is interpretted. In current uses, the number is interpretted as number of connections desired (1 for 1). In the case that duplex network connections are being reassembled an analyzed, the producer will return two streams (one for each direction) per connection. Again whether the connections are simplex or duplex must be agreed out of band.

Request Message (sent by client to server):


[1] Number of connections (0-255)


0 => 0 data messages requested, an optional status message can be returned
1 - 255 => return the respective number of data streams

Response Message (sent by server to client):

[1] The literal '#' or 0x23
[4] Metadata length m (0-PATH_MAX)
[m] Metadata
[1] The literal '|' or 0x7C
[4] Data length n (0-PATH_MAX)
[n] Data

All lengths are unsigned integers in network byte order. A metadata length of 0 indicates a server disconnect message.



*/



//Define some Macros for error logging
#define DEBUG(l,m...) { if (l <= debug_level) { LOG(LOG_DEBUG, m); } }
#define WARN(m...) {  LOG(LOG_WARNING, m); }
#define ERROR(m...) {  LOG(LOG_ERR, m); }
#define DIE(m...) { LOG(LOG_CRIT, m); exit(1); }
#define LOG(p,m...) { syslog(p, m); fprintf(stderr, m); fprintf(stderr, "\n"); }



//global vars,Defaults
//for getopt
extern int optind;
extern int opterr;
extern int optopt;
extern char *optarg;


char *temp_data_dir="/tmp";


uint16_t port;
unsigned char connections_per_batch = 2;

//max number of times to connect to ruminate server
int max_ruminate_connections = 0;
int ruminate_connection_counter = 0;

//rest are about connections (streams) transfer over protocol
int max_connections = 0;
unsigned long int connection_counter = 0;
int streams_per_connection = 1;
char *process_name;

int server_d;
char *host;
struct sockaddr_in server_addr;
struct hostent *hostent_p = NULL;

int debug_level = 0;
int pipeline_mode = 0;

int socket_status = 0;
//0 no sd yet
//1 socket succeeded, not yet connected
//2 connected, able to send data
//3 connection failed, reads/writes fail (need to close)


#define BLOCK_SIZE 4096




size_t fill_buffer(int desc, void *buf, size_t size)
{
    int num_read = 0;
    int num_read_total = 0;
    int num_left = size;
    
    while ((num_read = read(desc,(buf+num_read_total),num_left))>0)
    {
	DEBUG(480, "Read %i bytes from socket", num_read);
	num_read_total += num_read;
        num_left -= num_read;
	if (num_left == 0)
	{
	    break;
	}
    }
    return num_read_total;
}

size_t copy_stof(int sock_desc, char *filename, size_t size)
{
    char full_path[PATH_MAX];
    int file_desc;
    size_t size_to_copy;
    size_t size_copied;
    char data_buff[BLOCK_SIZE];
    
    
    
    strncpy(full_path, temp_data_dir, PATH_MAX);
    if (full_path[strlen(full_path)] != '/' )
    {
	strncat(full_path, "/", PATH_MAX);
    }
    strncat(full_path, filename, PATH_MAX);
    
    file_desc = creat(full_path, S_IRUSR ^ S_IWUSR ^ S_IRGRP ^ S_IWGRP ^ S_IROTH ^ S_IWOTH);
    if (file_desc < 0)
    {
	DIE("Couldn't open output file: %s", full_path);
    }
    DEBUG(400, "Opened File %s",full_path);
    
    //I don't think I can use senfile here because I'm reading from socket
    
    size_copied = 0;
    while( size_copied < size )
    {
	if ( (size - size_copied) > BLOCK_SIZE)
	{
	    size_to_copy = BLOCK_SIZE;
	} else
	{
	    size_to_copy = (size - size_copied);
	}
	
	if (fill_buffer(sock_desc, data_buff, size_to_copy) == size_to_copy)
	{
	    if (write(file_desc, data_buff, size_to_copy) != size_to_copy)
	    {
		DIE("Error writing to output file");
	    }
	    size_copied += size_to_copy;
	} else
	{
	    WARN("Couldn't read data from socket!");
	    break;
	}
	
    }
    
    close(file_desc);
    printf("%s\n",full_path);
    
    return size_copied;
}


//read a int from network. Do necessary translations
//returns 0 on failure, else on success
int read_uint32_n(int desc, uint32_t *target)
{
    uint32_t network_encoded;
    if ( fill_buffer(desc, (void *)&network_encoded, sizeof(uint32_t) ) == sizeof(uint32_t) )
    {
        *target = ntohl(network_encoded);
        DEBUG(450, "Read %x from socket which translates to %x", network_encoded, *target);
	return 1;
    } else
    {
        return 0;
    }
    
}


void print_usage()
{
	fprintf(stderr,"Usage: %s -h host -p port [ -bce count ] [ -D level ] [ -t dir ] [ -dlP ] \n", process_name);
	fprintf(stderr,
	    "	-h host		Hostname of server\n"
	    "	-p port		Server port\n"
	    "	-t dir		Dir in which to store streams (default: /tmp)\n"
	    "	-b count	Number of connections to download per batch (1-255, default: 2)\n"
	    "	-c count	Total number of connections to download (0 for infinity, default: 0)\n"
	    "	-D level	Set debug level (0-500, default: 0)\n"
	    "	-l 		Set output (metadata) to line buffering\n"
	    "	-P		Enable pipeline mode (extra request for connections is sent, default: disabled)\n"
	    "	-e count	Exit after count successful connections to server \n"
	    "	-d 		Duplex mode: server is providing both directional streams per connection, not just one (default: simplex mode)\n");
	    
    fprintf(stderr,"   \n\n");
	
}





int main (int argc, char **argv)
{
    uint32_t length;
    unsigned char connections_left;
    unsigned char marker;
    int streams_left;
    char metadata_buffer[PATH_MAX+1];
    int opt;
    process_name = argv[0];
    
    
    while ((opt = getopt(argc, argv, "c:h:p:b:t:dD:lPe:")) != -1)
    {
	switch (opt) 
	{
	    case 'c':
		max_connections = atoi(optarg);
		break;
	    case 'e':
		max_ruminate_connections = atoi(optarg);
		break;
	    case 'h':
		host = optarg;
		break;
	    case 't':
		temp_data_dir = optarg;
		break;
	    case 'p':
		port = atoi(optarg);
		break;
	    case 'b':
		connections_per_batch = (unsigned int) atoi(optarg);
		break;
            case 'd':
		streams_per_connection = 2;
		break;
	    case 'D':
		debug_level = atoi(optarg);
		break;
	    case 'l':
		//set STDOUT to line buffering
		#ifdef HAVE_SETLINEBUF
		    setlinebuf(stdout);
		#else
		    setvbuf(stdout, NULL, _IOLBF, 0);
		#endif
		DEBUG(100,"Ouput set to line buffering");
		break;            
	    case 'P':
		pipeline_mode = 1;
		DEBUG(100,"Enabled Pipeline mode");
		break;
	    default:
  		print_usage();
  		DIE("Invalid option %c", opt);
  		break;
  	}
    }
    
        
      
    //connect to socket
    hostent_p = gethostbyname(host);
    if (!hostent_p)
    {
        DIE("Error resolving host");
    }
  
    memset(&server_addr, 0, sizeof(server_addr));
    server_addr.sin_family = AF_INET;
    bcopy(hostent_p->h_addr, (char *)&server_addr.sin_addr, hostent_p->h_length);
    server_addr.sin_port = htons(port);


    

    while( ((connection_counter < max_connections ) || (max_connections <= 0)) && ((ruminate_connection_counter < max_ruminate_connections ) || (max_ruminate_connections <= 0)) )
    {
	if (socket_status == 3)
	{
	    close(server_d);
	    socket_status = 0;
	    streams_left = 0;
	}
	
	
	if (socket_status == 0)
	{
	    if ( (server_d = socket(AF_INET, SOCK_STREAM, 0)) < 0 ) 
	    {
		WARN("Error creating socket.");
		sleep(1);
		break;
	    } else
	    {
		socket_status = 1;
	    }    
	}
	
	if (socket_status == 1)
	{
	    if ( connect(server_d, (struct sockaddr *) &server_addr, sizeof(server_addr)) < 0 )
	    {
		DEBUG(250, "Failed to connect.")
		sleep(1);
		continue;
	    } else
	    {
		socket_status = 2;
	    }
	}
	
    
	DEBUG(50, "Connection established, downloading connecitons");
	ruminate_connection_counter++;
    
	if (pipeline_mode)
	{
	    
	    if (write(server_d, &connections_per_batch, 1) != 1)
	    {
		WARN("Couln't request connections");
		socket_status = 3;
		continue;
	    } else
	    {
		DEBUG(280, "Pipeline: Requesting %hhu connections", connections_per_batch);
	    }
	}
    
	while( (connection_counter < max_connections ) || (max_connections <= 0) )
	{
	    if (socket_status != 2)
	    {
		break;
	    }
	    
	    if ( ((max_connections - (connection_counter + connections_per_batch * pipeline_mode)) > connections_per_batch) || (max_connections <= 0) )
	    {
	        connections_left = connections_per_batch;
	        streams_left = connections_left * streams_per_connection;
	    } else
	    {
	        connections_left = (max_connections - (connection_counter + connections_per_batch * pipeline_mode));
	        streams_left = connections_left * streams_per_connection;
	    }
	            
	    
	    if (write(server_d, &connections_left, 1) != 1)
	    {
		WARN("Couln't request connections");
		socket_status = 3;
		continue;
	    } else
	    {
		DEBUG(280, "Requested %hhu connections", connections_left);
	    }
	    
	    for(; streams_left > 0; streams_left--)
	    {
	        connection_counter++;
	        DEBUG(300, "Reading connection %li",connection_counter);
		
		//check for start of message marker (#)
		if (read(server_d,&marker,sizeof(marker)) == sizeof(marker))
		{
		    if (marker != '#')
		    {
			WARN("Error reading start of message marker: %hhx", marker);
			socket_status = 3;
			break;
		    }
		} else
		{
		    WARN("Error reading start of message marker");
		    socket_status = 3;
		    break;
		}
		
		//read the size of metadata
	        if (read_uint32_n(server_d, &length))
	        {
		    if (length == 0)
		    {
			WARN("Server sent disconnect command");
			socket_status = 3;
			break;
		    }
		    
		    if (length > PATH_MAX)
		    {
		        WARN("Length metadata too long: %u", length);
			socket_status = 3;
			break;
		    } else
		    {
		        metadata_buffer[length] = '\0';
		        if ( fill_buffer(server_d, (void *)metadata_buffer, length ) != length )
		        {
			    WARN("Couldn't read metadata!");
			    socket_status = 3;
			    break;
		        }
		        
			//look for data marker
			
			if (read(server_d,&marker,sizeof(marker)) == sizeof(marker))
			{
			    if (marker != '|')
			    {
				WARN("Error reading start of data marker: %hhx", marker);
				socket_status = 3;
				break;
			    }
			} else
			{
			    WARN("Error reading start of data marker");
			    socket_status = 3;
			    break;
			}
			
			//now read the data (length first)
		        if (read_uint32_n(server_d, &length))
		        {
			    if (copy_stof(server_d, metadata_buffer, length) != length)
			    {
				WARN("Error reading data from socket");
				socket_status = 3;
				break;
			    }
			} else
			{
			    WARN("Error reading data length from socket");
			    socket_status = 3;
			    break;
			}
		    }
		} else
		{
		    WARN("Error reading metadata length from socket");
		    socket_status = 3;
		    break;
		}
	           
	    }   
	}
    }	
    DEBUG(50, "Done downloading connections, quiting");
    return 0;
}







