/***********************************************************************

 What This Is:      Interface to Parse Tree ADT
 Author:            Laura Deddens
 E-mail             lelo@cats.ucsc.edu
 Last Modified:     5 September 1996

 A "Parse Tree" is abstractly a root node with zero or more ordered
 children which are parse trees themselves.  Each parse tree has
 associated with it an (x,y) position, a string, n values numbered
 0 ... (n-1) with n > 0, and a right Sibling which is NULL if it has no
 parent or if it is the right most child of its parent otherwise the
 right sibling is the ParseTreeHdl of the parent's child which is
 just to the right of this one.

***********************************************************************/


/***********************************************************************
 The following defines the parse tree handle type.  Clients should
 declare (and initialize to NULL) one variable of type "ParseTreeHdl"
 for each parse tree they want to have.  After declaration (and
 initialization to NULL), the "CreateParseTree" procedure must be called
 before any operations can be performed.
***********************************************************************/
typedef struct ParseTreeStruct * ParseTreeHdl;


/*----------Creation and deletion procedures----------*/
/***********************************************************************
 This should be called once on each parse tree before any other
    operations are performed on that parse tree.
 Precondition:   tree is NULL.
                 numValues > 0.
                 sLength is the length of s.
 Postcondition:  tree will be initialized to be a parse tree with no
    children.  It will have numValues values associated with it each
    initialized to 0.  Its string will be s and its (x,y) position
    will be x and y.
***********************************************************************/
void CreateParseTree(ParseTreeHdl * tree, int numValues, char *s, int sLength, int x, int y);

/***********************************************************************
 This should be called once on each parse tree when it is no longer
    needed.  Its purpose is to free up the storage used by tree and all
    its children.
 Precondition:   tree has been initialized by "CreateParseTree"
 Postcondition:  tree is set to NULL
***********************************************************************/
void DestroyParseTree(ParseTreeHdl * tree);

/*----------Query functions----------*/
/***********************************************************************
 Precondition:   tree has been initialized by "CreateParseTree"
 Postcondition:  x and y are the (x,y) position associated with tree
***********************************************************************/
void GetParseTreePosition(ParseTreeHdl tree, int *x, int *y);

/***********************************************************************
 Precondition:   tree has been initialized by "CreateParseTree"
 Postcondition:  s is the string associated with tree and sLength is
    the length of s.
***********************************************************************/
void GetParseTreeString(ParseTreeHdl tree, char* *s, int * sLength);

/***********************************************************************
 Precondition:   tree has been initialized by "CreateParseTree"
                 which >= 0 and which < n where n is the value that
    numValues had when "CreateParseTree" was called on tree.
 Postcondition:  returns returns the which-th value associated with tree
***********************************************************************/
int GetParseTreeOrgValue(ParseTreeHdl tree, int which);

/***********************************************************************
 Precondition:   tree has been initialized by "CreateParseTree"
 Postcondition:  returns tree's left most child
***********************************************************************/
ParseTreeHdl GetLeftChild(ParseTreeHdl tree);

/***********************************************************************
 Precondition:   tree has been initialized by "CreateParseTree"
 Postcondition:  returns tree's right sibling
***********************************************************************/
ParseTreeHdl GetRightSib(ParseTreeHdl tree);


/*----------Manipulation procedures----------*/
/***********************************************************************
 Precondition:   tree and child have been initialized by
    "CreateParseTree", neither of them have a parent
                 nodeHeight > 0 and is the number of pixal that you want
    tree's y position to be above all of its children's y positions
                 lowestY is an upperbound on the y positions of all the
    nodes in tree and in child
 Postcondition:  tree has child as a new child positioned right most.
                 tree's x position has been modified so that it is
    centered over all it's children
                 tree's y position has been modified so that it is
    nodeHeight pixels above it's heighest child.
                 tree's old right most child now has child as its
    right sibling.
***********************************************************************/
void SetNewRightChild(ParseTreeHdl tree, ParseTreeHdl child, int nodeHeight, int lowestY);

/***********************************************************************
 Precondition:   tree has been initialized by "CreateParseTree"
                 which >= 0 and which < n where n is the value that
    numValues had when "CreateParseTree" was called on tree.
 Postcondition:  the which-th value associated with tree is now
    orgValue
***********************************************************************/
void SetParseTreeOrgValue(ParseTreeHdl tree, int orgValue, int which);
