/***********************************************************************

 What This Is:      Interface to ParseTreeHdl List ADT
 Author:            Laura Deddens
 Email:             lelo@cats.ucsc.edu
 Last Modified:     5 September 1996

 An "ParseTreeHdl List" is abstractly a sequence of ParseTreeHdls and a
    flag.  One of the ParseTreeHdls in the sequence may be associated
    with the flag.  If elements are inserted/deleted from the sequence,
    the flag will maintain its associated I.e. if the flag was
    unassociated then it will still be unassociated; if the flag was
    associated with an ParseTreeHdl then it will still be associated
    with that same ParseTreeHdl. (Unless the flagged ParseTreeHdl is
    deleted, in which case the flag will no longer be associated with
    any of the ParseTreeHdls in the sequence, that is it will be
    unassociated).

Note:   You must include parseTree.h before including this file.

***********************************************************************/


/***********************************************************************
 The following defines the ParseTreeHdl list handle type.  Clients
 should declare (and initialize to NULL) one variable of type
 "ParseTreeListHdl" for each sequence they want to have.  After
 declaration (and initialization to NULL), the "CreateParseTreeList"
 procedure must be called before any operations can be performed.
***********************************************************************/
typedef struct ParseTreeListStruct * ParseTreeListHdl;


/*----------Creation and deletion procedures----------*/
/***********************************************************************
 This should be called once on each list before any other operations
    are performed on that list.
 Precondition:   theList is NULL.
 Postcondition:  theList will be initialized to an empty sequence.
***********************************************************************/
void CreateParseTreeList(ParseTreeListHdl * theList);

/***********************************************************************
 This should be called once on each list when the list is no longer
    needed.  Its purpose is to free up the storage used by theList.
 Precondition:   theList has been initialized by "CreateParseTreeList"
 Postcondition:  theList is set to NULL
***********************************************************************/
void DestroyParseTreeList(ParseTreeListHdl * theList);


/*----------Query functions----------*/
/***********************************************************************
 Precondition:   theList has been initialized by "CreateParseTreeList"
 Postcondition:  returns true if the sequence is empty and false
    otherwise
***********************************************************************/
int IsEmpty(ParseTreeListHdl theList);

/***********************************************************************
 Precondition:   theList has been initialized by "CreateParseTreeList"
                 not IsEmpty
 Postcondition:
    "AtFirst" returns true if the first ParseTreeHdl in the sequence is
       flagged and false otherwise.
    "AtLast" returns true if the last ParseTreeHdl in the sequence is
       flagged and false otherwise.
***********************************************************************/
int AtFirst(ParseTreeListHdl theList);
int AtLast(ParseTreeListHdl theList);

/***********************************************************************
 Precondition:   theList has been initialized by "CreateParseTreeList"
 Postcondition:  returns true if there is a flagged ParseTreeHdl in the
    sequence and false otherwise
***********************************************************************/
int ExistsFlagged(ParseTreeListHdl theList);

/***********************************************************************
 Precondition:   theList has been initialized by "CreateParseTreeList"
                 ExistsFlagged is true
 Postcondition:  returns the value of the flagged ParseTreeHdl.
***********************************************************************/
ParseTreeHdl FlaggedParseTreeHdl(ParseTreeListHdl theList);

/***********************************************************************
 Precondition:   theList has been initialized by "CreateParseTreeList"
 Postcondition:  returns the number of elments in theList
***********************************************************************/
int NumberOfElements(ParseTreeListHdl theList);


/*----------Manipulation procedures----------*/
/***********************************************************************
 These procedures associate the flag with one of the ParseTreeHdls in the
    sequence represented by theList.
 Precondition:
    for all four procedures:
       theList has been initialized by "CreateParseTreeList"
       not IsEmpty
    for only both "MovePrev" and "MoveNext":
       ExistsFlagged is true
 Postcondition:
    "MoveFirst": flags the first ParseTreeHdl (i.e. the flag moves to
       the first ParseTreeHdl)
    "MoveLast":  flags the last ParseTreeHdl (i.e. the flag moves the
        the last ParseTreeHdl)
    "MovePrev":  flags the ParseTreeHdl which precedes the currently
       flagged ParseTreeHdl (i.e. the flag moves over one to the "left")                 If AtFirst was true before calling "MovePrev" then the
       flag will become unassociated. (i.e. ExistsFlagged will be false)
    "MoveNext":  flags the ParseTreeHdl which follows the currently
       flagged ParseTreeHdl (i.e. the flag moves over one to the
       "right")
                 If AtFirst was true before calling "MoveNext" then the
       flag will become unassociated. (i.e. ExistsFlagged will be false)
***********************************************************************/
void MoveFirst(ParseTreeListHdl theList);
void MoveLast(ParseTreeListHdl theList);
void MovePrev(ParseTreeListHdl theList);
void MoveNext(ParseTreeListHdl theList);

/***********************************************************************
 These procedures insert additional ParseTreeHdls into the sequence of
    ParseTreeHdls represented by theList.
 Precondition:
    for all three procedures:
       theList has been initialized by "CreateParseTreeList"
    for "AddAfterFlagged" only:
       ExistsFlagged is true
 Postcondition:
    "AddFirst": adds its ParseTreeHdl argument to the front of the
       sequence
    "AddLast": adds its ParseTreeHdl argument to the rear of the
       sequence
    "AddAfterFlagged": adds its ParseTreeHdl argument immediately after
       the ParseTreeHdl which is flagged.
 Note:           None of these fuctions change which ParseTreeHdl is
    flagged.  If the the flag was unassociated then it will still be
    unassociated.
***********************************************************************/
void AddFirst(ParseTreeListHdl theList, ParseTreeHdl theData);
void AddLast(ParseTreeListHdl theList, ParseTreeHdl theData);
void AddAfterFlagged(ParseTreeListHdl theList, ParseTreeHdl theData);

/***********************************************************************
 These procedures delete ParseTreeHdls from the sequence of ParseTreeHdls
    represented by theList.
 Precondition:
    for all three procedures:
       theList has been initialized by "CreateParseTreeList"
       not Empty
    for "DeleteFlagged":
       ExistsFlagged is true
 Postcondition:  
    "DeleteFirst":
       deletes the first ParseTreeHdl in the sequence.
       If AtFirst was true before the deletion, then the flag will be
          unassociated.
    "DeleteLast":
       deletes the last ParseTreeHdl in the sequence.
       If AtLast was true before the deleteion, then the flag will be
          unassociated.
    "DeleteFlagged":
       deletes the ParseTreeHdl in the sequence associated with the flag
          and the flag will become unassociated.
***********************************************************************/
void DeleteFirst(ParseTreeListHdl theList);
void DeleteLast(ParseTreeListHdl theList);
void DeleteFlagged(ParseTreeListHdl theList);

/*----------Debugging procedures----------*/
/***********************************************************************
 This funtion is for debugging only.  It prints the state of theList
    to stderr.
***********************************************************************/
void DumpList(ParseTreeListHdl theList);
